<?php

namespace App\Services;

use App\Models\PaymentLog;
use App\Models\User;
use App\Models\UserSubscription;
use App\Models\Plan;
use App\Notifications\PushNotification;
use App\Services\Common\BaseService;
use Carbon\Carbon;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeCheckoutSession;
use Stripe\Subscription;
use Stripe\Customer;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Notification;

class UserSubscriptionService extends BaseService
{
    public function __construct()
    {
        // Initialize Stripe with the secret key
        Stripe::setApiKey(config('stripe.secret_key'));
    }

    /**
     * Create a subscription for the user using Stripe Checkout
     *
     * @param int $userId
     * @param int $planId
     * @return \Stripe\Checkout\Session
     */
    public function createCheckoutSession($userId, $planId)
    {
        // Retrieve the user
        $user = User::findOrFail($userId);

        // Retrieve the plan the user is subscribing to
        $plan = Plan::findOrFail($planId);

        // Create a Stripe customer if the user doesn't have one
        if (!$user->stripe_customer_id) {
            // Create a new Stripe customer
            $customer = Customer::create([
                'email' => $user->email,
                'metadata' => [
                    'user_id' => $user->id, // Add user_id to metadata
                ],
            ]);

            // Save the customer ID in the database
            $user->update(['stripe_customer_id' => $customer->id]);
        } else {
            // Retrieve the existing Stripe customer
            $customer = Customer::retrieve($user->stripe_customer_id);

            // Check if the user_id in metadata is different from the current user_id
            if ($customer->metadata->user_id != $user->id) {
                // Update the metadata with the correct user_id
                $customer->metadata = [
                    'user_id' => $user->id, // Update user_id in metadata
                ];
                $customer->save();  // Save the updated customer
            }
        }

        // Create the Stripe Checkout session
        $checkoutSession = StripeCheckoutSession::create([
            'payment_method_types' => ['card'],
            'metadata' => [
                'service_message' => json_encode([
                    'price' => $plan->stripe_price_id,
                ])
            ],
            'customer' => $customer->id,  // Use the created customer with metadata
            'line_items' => [
                [
                    'price' => $plan->stripe_price_id, // Stripe Price ID for the plan
                    'quantity' => 1,
                ],
            ],
            'mode' => 'subscription',  // Subscription mode
            'success_url' => route('subscription.success') . '?user_id=' . $user->id,
            'cancel_url' => route('subscription.cancel'),
        ]);
        Cache::set("stripe_subscription_checkout_id_$user->id", $checkoutSession->id);
        // Return the checkout session object
        return $checkoutSession;
    }


    public function handleCheckoutSessionSuccess($request)
    {
        try {
            $user_id = $request->query('user_id');
            if (Cache::has("stripe_subscription_checkout_id_$user_id")) {
                $sessionId = Cache::get("stripe_subscription_checkout_id_$user_id");
                $session = StripeCheckoutSession::retrieve($sessionId);
                $customer = Customer::retrieve($session->customer);
                $user = User::find($customer->metadata->user_id);
                // Get the user from the Stripe customer
                $service_message = json_decode($session->metadata->service_message);

                // Retrieve the subscription details from the session
                $subscriptionId = $session->subscription;
                $priceId = $service_message->price;

                // Find the plan associated with this price ID
                $plan = Plan::where('stripe_price_id', $priceId)->first();

                // Store subscription details in the local database
                $userSubscription = UserSubscription::create([
                    'user_id' => $customer->metadata->user_id, // Store user_id if present in metadata
                    'plan_id' => $plan->id,
                    'stripe_customer_id' => $customer->id,
                    'stripe_subscription_id' => $subscriptionId,
                    'stripe_price_id' => $priceId,
                    'auto_renew' => true, // Auto-renew is enabled by default
                    'ends_at' => null, // Subscription is active until canceled
                ]);
                $paymentLog = new PaymentLog();
                $paymentLog->paymentable_type = "App\Models\UserSubscription";
                $paymentLog->paymentable_id = $userSubscription->id;
                $paymentLog->stripe_checkout_id = $session->id;
                $paymentLog->currency = "usd";
                $paymentLog->amount = $plan->amount;
                $paymentLog->user_id = $customer->metadata->user_id;
                $paymentLog->save();

                $title = $user->first_name . " " . $user->last_name . " subscribed $plan->name plan!";
                $body = "User ID: " . $user->id;
                $data = [
                    "user" => $user,  // Optionally, you can pass just the ID or a limited set of user info
                    "title" => $title,
                    "body" => $body,
                ];
                $admin = User::where('type', 'admin')->first();
                sendNotification($title, $body, $data, $user->id, $admin);
                $token = $user->device_token;
                if ($token && $token != "undefined") {
                    $firebaseService = new FirebaseService();
                    $firebaseService->sendNotification($token, "Payment Successful", "Subscription Payment Successful");
                }
                // Notification::route('firebase', $user->device_token)->notify(new PushNotification($title, $body));
                return view('front.payment-response', [
                    'status' => 'success',
                    'title' => 'Payment Successfull!',
                    'message' => 'Payment Successfull !',
                ]);
            }
        } catch (\Exception $e) {
            Log::error("Error handling Checkout session success: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Cancel a user's subscription on Stripe.
     *
     * @param int $userId
     * @return string
     */
    public function cancelSubscription($userId)
    {
        // Retrieve the user's subscription
        $userSubscription = UserSubscription::where('user_id', $userId)->first();

        if (!$userSubscription) {
            return response()->json(['message' => 'No active subscription found'], 404);
        }

        // Retrieve the Stripe subscription
        $subscription = Subscription::retrieve($userSubscription->stripe_subscription_id);

        // Cancel the subscription on Stripe
        $subscription->cancel();

        // Update the subscription status in the local database
        $userSubscription->update(['ends_at' => now()]);

        return 'Subscription canceled successfully';
    }

    public function userSubscriptions($userId)
    {
        // Retrieve the user's subscription
        $userSubscription = UserSubscription::where('user_id', $userId)
            ->where('created_at', '<=', Carbon::now())
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'));

                if (request()->filled('toDate')) {
                    $q->whereDate('created_at', '<=', request('toDate'));
                }
            })
            ->when(request()->filled('type'), function ($q) {
                $q->whereHas('plan', function ($query) {
                    $query->where('interval', request('type')); // filter by interval from request
                });
            })
            ->with('plan')
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);

        return $userSubscription;
    }


    public function userCurrentSubscription($userId)
    {
        // Retrieve the user's subscription
        $userSubscription = UserSubscription::where('user_id', $userId)
            ->with('plan') // Eager load the plan relation
            ->get()
            ->filter(function ($subscription) {
                $interval = $subscription->plan?->interval ?? 'month'; // e.g. 'month', 'week', 'year'
    
                try {
                    $expiresAt = Carbon::parse($subscription->created_at)->add(1, $interval);
                    return $expiresAt->gte(now());
                } catch (\Exception $e) {
                    return false; // in case interval is invalid
                }
            })
            ->sortByDesc('created_at')
            ->first();
        return $userSubscription;
    }
}
