<?php
namespace App\Services;

use App\Http\Resources\Admin\UserListingResource;
use App\Models\TimeSlot;
use App\Models\User;
use App\Models\WeeklyTimeSlot;
use App\Services\Common\BaseService;
use Carbon\Carbon;
class TimeSlotService extends BaseService
{
    // public function store($request)
    // {
    //     // Create the new time slot in the database
    //     $timeSlot = TimeSlot::create([
    //         'consultant_id' => $request->consultant_id,
    //         'date' => $request->date,
    //         'time' => $request->time,
    //         'status' => 'available', // Default status
    //     ]);
    //     return $timeSlot;
    // }
    // public function create($request)
    // {
    //     dd($request);
    //     $consultant = User::findOrFail($request->consultant_id);

    //     // Iterate over each day's timeslot and store it
    //     foreach ($request->timeslots as $timeslot) {
    //         $day = $timeslot['day'];
    //         $time = $timeslot['time'];
    //         $status = $timeslot['status'];

    //         // Get the date for the specified day of the week
    //         $date = Carbon::parse("next $day");

    //         // Check if the timeslot for this day and time already exists, and update if necessary
    //         $existingSlot = Timeslot::where('consultant_id', $consultant->id)
    //             ->where('date', $date->format('Y-m-d'))
    //             ->where('time', $time)
    //             ->first();

    //         if ($existingSlot) {
    //             // Update the existing timeslot status
    //             $existingSlot->status = $status;
    //             $existingSlot->save();
    //         } else {
    //             // Create a new timeslot
    //             Timeslot::create([
    //                 'consultant_id' => $consultant->id,
    //                 'date' => $date->format('Y-m-d'),
    //                 'time' => $time,
    //                 'status' => $status
    //             ]);
    //         }
    //     }

    //     return 'Timeslots updated successfully';
    // }
    // public function create($request)
    // {
    //     // If validation fails, it will automatically return a response with errors

    //     // Proceed with the logic to process the valid data
    //     $consultant = User::findOrFail($request->consultant_id);

    //     // Filter enabled days and perform the rest of the logic as shown before
    //     $timeslots = collect($request->timeslots)->filter(function ($timeslot) {
    //         return $timeslot['is_enabled'];
    //     });

    //     // Prepare for batch checking existing slots
    //     $existingSlots = Timeslot::where('consultant_id', $consultant->id)
    //         ->whereIn('date', $timeslots->pluck('day')->map(function ($day) {
    //             // Get the current day name (e.g., "Monday", "Tuesday")
    //             $currentDay = Carbon::now()->format('l'); // Today's day of the week (e.g., "Monday")

    //             // If today's day matches the timeslot's day, use today's date
    //             if ($currentDay === $day) {
    //                 return Carbon::now()->format('Y-m-d'); // Use today's date
    //             } else {
    //                 // Otherwise, get the "next" occurrence of the desired day
    //                 return Carbon::parse("next $day")->format('Y-m-d');
    //             }
    //         }))
    //         ->get()
    //         ->groupBy('date');

    //     // Proceed with processing timeslots and slots as before
    //     foreach ($timeslots as $timeslot) {
    //         if (isset($timeslot['slots'])) {
    //             $currentDay = Carbon::now()->format('l');

    //             foreach ($timeslot['slots'] as $slot) {
    //                 // If today's day is the same as the desired day, use today's date
    //                 if ($currentDay === $timeslot['day']) {
    //                     $date = Carbon::now(); // Use today's date
    //                 } else {
    //                     // Otherwise, get the "next" occurrence of the desired day (e.g., "next Monday")
    //                     $date = Carbon::parse("next {$timeslot['day']}");
    //                 }

    //                 $formattedTime = Carbon::createFromFormat('H:i', $slot['time'])->format('H:i:s');
    //                 $status = $slot['status'];
    //                 // Check     if the date key exists in the $existingSlots collection
    //                 $existingSlotCollection = $existingSlots->get($date->format('Y-m-d'));
    //                 // If the collection for the specific date exists, find the slot by time
    //                 if ($existingSlotCollection) {
    //                     $existingSlot = $existingSlotCollection->firstWhere('time', $formattedTime);
    //                 } else {
    //                     // If no slots exist for the date, set existingSlot to null
    //                     $existingSlot = null;
    //                 }

    //                 if ($existingSlot) {
    //                     // If the existing slot's status is not "BOOKED", update its status
    //                     if ($existingSlot->status != TimeSlot::BOOKED) {
    //                         $existingSlot->status = $status;
    //                         $existingSlot->save();
    //                     }
    //                 } else {
    //                     // If no existing slot is found, create a new one
    //                     Timeslot::create([
    //                         'consultant_id' => $consultant->id,
    //                         'date' => $date->format('Y-m-d'),
    //                         'time' => $formattedTime,
    //                         'status' => $status
    //                     ]);
    //                 }
    //             }
    //         }
    //     }

    //     return 'Timeslots updated successfully';
    // }
    public function create($request)
    {
        $consultant = User::findOrFail($request->consultant_id);

        // 1) Update recurring weekly templates (new behaviour)
        $this->upsertWeeklyTemplates($consultant->id, $request->timeslots);

        // 2) Keep existing behaviour for now: generate concrete slots for the current week
        //    so that existing booking flow using TimeSlot & time_slot_id continues to work.

        // Filter only enabled days
        $timeslots = collect($request->timeslots)->filter(fn($timeslot) => $timeslot['is_enabled']);

        // Prepare day->date map for current week (Mon to Sun)
        $currentWeekMap = collect();
        $startOfWeek = Carbon::now()->startOfWeek(); // Monday

        for ($i = 0; $i < 7; $i++) {
            $date = $startOfWeek->copy()->addDays($i);
            $currentWeekMap->put($date->format('l'), $date->format('Y-m-d')); // 'Monday' => '2025-04-14'
        }

        // Get dates for all selected days in the current week
        $selectedDates = $timeslots->pluck('day')->map(fn($day) => $currentWeekMap->get($day))->filter();

        // Fetch existing slots grouped by date
        $existingSlots = TimeSlot::where('consultant_id', $consultant->id)
            ->whereIn('date', $selectedDates)
            ->get()
            ->groupBy('date');

        foreach ($timeslots as $timeslot) {
            if (!isset($timeslot['slots'])) {
                continue;
            }

            $date = $currentWeekMap->get($timeslot['day']);
            if (!$date) {
                continue;
            }

            foreach ($timeslot['slots'] as $slot) {
                $formattedTime = Carbon::createFromFormat('H:i', $slot['time'])->format('H:i:s');
                $status = $slot['status'];

                // Check if slot already exists in the DB for that date/time
                $existingSlotCollection = $existingSlots->get($date);
                $existingSlot = $existingSlotCollection?->firstWhere('time', $formattedTime);

                // Skip if already exists (don't override existing bookings)
                if ($existingSlot) {
                    continue;
                }

                // Prevent duplicate insertions in current request
                $existingSlots[$date] = $existingSlots[$date] ?? collect();
                if ($existingSlots[$date]->contains('time', $formattedTime)) {
                    continue;
                }

                // Add to memory collection to prevent re-processing
                $existingSlots[$date]->push((object) [
                    'time' => $formattedTime,
                    'status' => $status,
                ]);

                TimeSlot::create([
                    'consultant_id' => $consultant->id,
                    'date' => $date,
                    'time' => $formattedTime,
                    'status' => $status,
                ]);
            }
        }

        return 'Timeslots updated successfully for current week.';
    }

    /**
     * Upsert recurring weekly templates based on the timeslots payload.
     *
     * Structure:
     * timeslots: [
     *   {
     *     day: "Monday",
     *     is_enabled: true,
     *     slots: [{ time: "10:00", status: "available" }, ...]
     *   },
     *   ...
     * ]
     */
    protected function upsertWeeklyTemplates(int $consultantId, array $timeslots): void
    {
        $enabledDays = collect($timeslots)->filter(fn($t) => $t['is_enabled']);
        $disabledDays = collect($timeslots)->reject(fn($t) => $t['is_enabled']);

        // Remove templates for disabled days
        if ($disabledDays->isNotEmpty()) {
            $days = $disabledDays->pluck('day')->all();
            WeeklyTimeSlot::where('consultant_id', $consultantId)
                ->whereIn('day_of_week', $days)
                ->delete();
        }

        // Upsert templates for enabled days
        foreach ($enabledDays as $timeslot) {
            if (empty($timeslot['slots'])) {
                continue;
            }

            $day = $timeslot['day'];

            foreach ($timeslot['slots'] as $slot) {
                $time = Carbon::createFromFormat('H:i', $slot['time'])->format('H:i:s');
                $status = $slot['status'] ?? 'available';

                WeeklyTimeSlot::updateOrCreate(
                    [
                        'consultant_id' => $consultantId,
                        'day_of_week' => $day,
                        'time' => $time,
                    ],
                    [
                        'status' => $status,
                    ]
                );
            }
        }
    }


    public function getConsultantTimeslotsByDate($consultantId, $date)
    {
        $consultant = User::findOrFail($consultantId);
        $date = Carbon::parse($date)->startOfDay(); // Normalize to start of day

        // Return empty collection if date is in the past
        if ($date->lt(Carbon::today())) {
            return collect();
        }

        // Ensure concrete TimeSlot rows exist for this date based on weekly templates
        $dayName = $date->format('l'); // Monday, Tuesday, ...

        $templates = WeeklyTimeSlot::where('consultant_id', $consultant->id)
            ->where('day_of_week', $dayName)
            ->where('status', 'available')
            ->get();

        foreach ($templates as $template) {
            TimeSlot::firstOrCreate(
                [
                    'consultant_id' => $consultant->id,
                    'date' => $date->toDateString(),
                    'time' => $template->time,
                ],
                [
                    'status' => $template->status,
                ]
            );
        }

        $query = Timeslot::where('consultant_id', $consultant->id)
            ->whereDate('date', $date)
            ->where('status', 'available');

        // If date is today, filter out past times
        if ($date->isToday()) {
            $currentTime = Carbon::now()->format('H:i:s');
            $query->whereTime('time', '>', $currentTime);
        }

        $timeslots = $query->orderBy('time')->get();

        $formattedTimeslots = $timeslots->map(function ($timeslot) {
            $start = Carbon::parse($timeslot->time)->format('h:i A');
            $end = Carbon::parse($timeslot->time)->addHour()->format('h:i A');

            return [
                'id' => $timeslot->id,
                'date' => Carbon::parse($timeslot->date)->format('Y-m-d'),
                'time' => Carbon::parse($timeslot->time)->format('H:i'),
                'time_range' => "{$start} - {$end}",
                'status' => $timeslot->status,
                'created_at' => $timeslot->created_at,
            ];
        });

        return $formattedTimeslots;
    }

    public function getConsultantTimeslotsByWeek($consultantId)
    {
        $consultant = User::findOrFail($consultantId);

        // Use recurring weekly templates instead of concrete dates
        $templates = WeeklyTimeSlot::where('consultant_id', $consultant->id)
            ->orderBy('day_of_week')
            ->orderBy('time')
            ->get()
            ->groupBy('day_of_week');

        // Build a fixed Monday–Sunday structure
        $daysOfWeek = [
            'Monday',
            'Tuesday',
            'Wednesday',
            'Thursday',
            'Friday',
            'Saturday',
            'Sunday',
        ];

        $weekDays = collect();

        foreach ($daysOfWeek as $dayName) {
            $slots = $templates->get($dayName, collect());

            $formattedSlots = $slots->map(function (WeeklyTimeSlot $slot) {
                $start = Carbon::parse($slot->time)->format('h:i A');
                $end = Carbon::parse($slot->time)->addHour()->format('h:i A');

                return [
                    'id' => $slot->id,
                    'time_range' => "{$start} - {$end}",
                    'from' => $start,
                    'to' => $end,
                    'status' => $slot->status,
                    'created_at' => $slot->created_at,
                ];
            });

            $weekDays->push([
                'day_name' => $dayName,
                'slots' => $formattedSlots,
            ]);
        }

        return [
            'week_pattern' => 'Recurring weekly slots',
            'days' => $weekDays,
        ];
    }
    public function getAvailableEvents($consultant_id)
    {
        // Get current date and time using Carbon
        $currentDateTime = Carbon::now();

        // Fetch available time slots from the database, excluding past slots
        $slots = TimeSlot::where('consultant_id', $consultant_id)
            ->where('status', 'available')
            ->where(function ($query) use ($currentDateTime) {
                $query->whereDate('date', '>', $currentDateTime->toDateString()) // Future dates
                    ->orWhere(function ($query) use ($currentDateTime) {
                        $query->whereDate('date', $currentDateTime->toDateString()) // Today
                            ->whereTime('time', '>=', $currentDateTime->toTimeString()); // Current or future time
                    });
            })
            ->get();

        // Transform the time slots into event data
        $events = $slots->map(function ($slot) {
            // Combine the date and time to create a start datetime
            $startTime = Carbon::createFromFormat('Y-m-d H:i:s', $slot->date . ' ' . $slot->time);

            // End time is 1 hour after the start time
            $endTime = $startTime->copy()->addHour();

            // Return the event data in the required format
            return [
                'title' => "Available: " . $slot->time,  // Title with the time
                'start' => $startTime->toIso8601String(), // Start time in ISO 8601 format
                'end' => $endTime->toIso8601String(), // End time in ISO 8601 format
                'allDay' => false, // Not an all-day event
                'resource' => $slot->id, // Slot ID
            ];
        });

        return ['timeslots' => $events, 'user' => User::find($consultant_id)];
    }

    /**
     * Resolve or create a concrete TimeSlot for a given consultant, date and time
     * based on the recurring WeeklyTimeSlot templates.
     */
    public function resolveSlotFromTemplate(int $consultantId, string $date, string $time): TimeSlot
    {
        $consultant = User::findOrFail($consultantId);

        $dateObj = Carbon::parse($date)->startOfDay();
        if ($dateObj->lt(Carbon::today())) {
            throw new \App\Exceptions\BadRequestException('You cannot book an appointment in the past.');
        }

        $dayName = $dateObj->format('l'); // Monday, Tuesday, ...
        $timeFormatted = Carbon::createFromFormat('H:i', $time)->format('H:i:s');

        // Ensure there is a weekly template for this day+time
        $template = WeeklyTimeSlot::where('consultant_id', $consultant->id)
            ->where('day_of_week', $dayName)
            ->where('time', $timeFormatted)
            ->where('status', 'available')
            ->first();

        if (! $template) {
            throw new \App\Exceptions\BadRequestException('Selected time is not available for this consultant.');
        }

        // Find or create the concrete slot for that date/time
        $slot = TimeSlot::firstOrCreate(
            [
                'consultant_id' => $consultant->id,
                'date' => $dateObj->toDateString(),
                'time' => $timeFormatted,
            ],
            [
                'status' => $template->status,
            ]
        );

        if ($slot->status === TimeSlot::BOOKED) {
            throw new \App\Exceptions\BadRequestException('This timeslot has already been booked.');
        }

        // If today, ensure the time is in the future
        if ($dateObj->isToday()) {
            $currentTime = Carbon::now()->format('H:i:s');
            if ($slot->time <= $currentTime) {
                throw new \App\Exceptions\BadRequestException('You cannot book a timeslot in the past.');
            }
        }

        return $slot;
    }
}
