<?php

namespace App\Services;

use App\Http\Resources\Appointment\AppointmentListingResource;
use App\Models\Appointment;
use App\Models\Category;
use App\Models\ConsultantInventory;
use App\Models\SubCategory;
use App\Services\Common\BaseService;

/**
 * Class JsonResponseService
 * @package App\Services
 */
class SubCategoryService extends BaseService
{
    public function create($request)
    {
        $subCategory = new SubCategory();
        $subCategory->name = $request->name;
        $subCategory->status = $request->status;
        $subCategory->quantity = $request->quantity;
        $subCategory->category_id = $request->category_id;
        $subCategory->save();
        return $subCategory;
    }

    public function update($id, $request)
    {
        $subCategory = SubCategory::findOrFail($id);
        $subCategory->name = $request->name;
        $subCategory->status = $request->status;
        $subCategory->quantity = $request->quantity;
        $subCategory->category_id = $request->category_id;
        $subCategory->save();
        return $subCategory;
    }

    public function delete($id)
    {
        $subCategory = SubCategory::findOrFail($id);
        $subCategory->delete();
        return 'Sub Category deleted successfully';
    }

    public function status($id)
    {
        $subCategory = SubCategory::findOrFail($id);
        $subCategory->status = ($subCategory->status == 1) ? 0 : 1;
        $subCategory->save();
        return $subCategory;
    }

    public function consultants($id)
    {
        $users = ConsultantInventory::with('user')
            ->where('sub_category_id', $id)
            ->get()
            ->pluck('user')
            ->unique()
            ->filter()
            ->values();
        return $users;
    }

    public function details($id, $user_id)
    {
        $query = ConsultantInventory::with('appointment')
            ->where('sub_category_id', $id)
            ->where('user_id', $user_id);

        // Total quantity and bookings
        $totalQuantity = $query->sum('quantity');
        $totalBookings = $query->count();
        $ids = $query->pluck("appointment_id")->toArray();
        $appointments = Appointment::whereIn('id', $ids)->when(request()->filled('search'), function ($q) {
            $q->where(function ($q) {
                $q->where('id', 'like', '%' . request("search") . '%');
            });
        })->paginate($this->pagination);
        return [
            'total_quantity' => $totalQuantity,
            'total_bookings' => $totalBookings,
            'appointments' => $appointments,
        ];
    }

    public function list()
    {
        $subCategories = SubCategory::when(request()->filled('status') && request('status') != 'all', function ($q) {
            $q->whereStatus(request('status'));
        })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('from'), function ($q) {
                $q->whereDate('created_at', '>=', request('from'))
                    ->whereDate('created_at', '<=', request('to'));
            })
            ->orderBy('id', 'desc') // Then order by id in descending order
            ->paginate($this->pagination);

        return $subCategories;
    }

    public function listActive()
    {
        $subCategories = SubCategory::where('status', 1)->when(request()->filled('status') && request('status') != 'all', function ($q) {
            $q->whereStatus(request('status'));
        })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('from'), function ($q) {
                $q->whereDate('created_at', '>=', request('from'))
                    ->whereDate('created_at', '<=', request('to'));
            })
            ->orderBy('id', 'desc') // Then order by id in descending order
            ->paginate($this->pagination);

        return $subCategories;
    }

    public function listActiveWithoutPagination()
    {
        $subCategories = SubCategory::where('status', 1)
            ->orderBy('id', 'desc') // Then order by id in descending order
            ->get();

        return $subCategories;
    }

    public function listActiveWithoutPaginationWithrespectToCategory($id)
    {
        $subCategories = SubCategory::where('category_id', $id)->where('status', 1)
            ->orderBy('id', 'desc') // Then order by id in descending order
            ->get();

        return $subCategories;
    }
}
