<?php


namespace App\Services;

use App\Models\Appointment;
use App\Models\Chat;
use App\Models\DeliveryCharges;
use App\Models\Order;
use App\Models\OrderAddress;
use App\Models\OrderDetail;
use App\Models\PaymentLog;
use App\Models\Product;
use App\Models\Shipment;
use App\Models\TimeSlot;
use App\Models\User;
use App\Notifications\OrderNotification;
use App\Notifications\PushNotification;
use App\Notifications\SendAdminNotification;
use App\Notifications\SendConfirmShipmentNotificationToGuestUser;
use Illuminate\Support\Facades\Notification;
use Stripe\Stripe;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class StripeService
{
    protected $stripe;

    public function __construct()
    {
        $this->stripe = new \Stripe\StripeClient(config('stripe.secret_key'));
    }

    public function createCustomer($name, $email)
    {
        return $this->stripe->customers->create([
            'name' => $name,
            'email' => $email,
        ]);
    }

    public function createCheckoutSession($amount, $user_id, $appointment_id, $chat_id)
    {
        $line_items[] = [
            'quantity' => 1, // Single delivery charge
            'price_data' => [
                'unit_amount' => (int) ($amount * 100), // Convert delivery charge to cents
                'currency' => 'usd',
                'product_data' => [
                    'name' => "Appointment Charges", // Name for the delivery charge
                ]
            ]
        ];
        $session = $this->stripe->checkout->sessions->create([
            'payment_method_types' => ['card'],
            'metadata' => [
                'service_message' => json_encode([
                    "user_id" => $user_id,
                    "appointment_id" => $appointment_id,
                    "unit_amount" => $amount,
                    "chat_id" => $chat_id,
                ])
            ],
            'line_items' =>
                $line_items,
            'mode' => 'payment',
            'success_url' => route('checkout.success') . '?user_id=' . $user_id,
            'cancel_url' => route('checkout.cancel'),
        ]);
        Cache::set("stripe_checkout_id_$user_id", $session->id);
        return $session;
    }

    public function success($request)
    {
        $user_id = $request->query('user_id');
        if (Cache::has("stripe_checkout_id_$user_id")) {
            $stripe_checkout_id = Cache::get("stripe_checkout_id_$user_id");
            $intenet_response = $this->stripe->checkout->sessions->retrieve($stripe_checkout_id, []);
            $service_message = json_decode($intenet_response->metadata->service_message);
            $user = User::find($service_message->user_id);
            if ($intenet_response->payment_status == "paid") {
                DB::beginTransaction();
                Cache::forget('stripe_checkout_id_$user_id');
                $paymentLog = new PaymentLog();
                $paymentLog->paymentable_type = "App\Models\Appointment";
                $paymentLog->paymentable_id = $service_message->appointment_id;
                $paymentLog->stripe_checkout_id = $stripe_checkout_id;
                $paymentLog->currency = "usd";
                $paymentLog->amount = $service_message->unit_amount;
                $paymentLog->user_id = $service_message->user_id;
                $paymentLog->save();
                $appointment = Appointment::where('id', $service_message->appointment_id)->first();
                $appointment->is_paid = 1;
                $appointment->amount = $service_message->unit_amount;
                $appointment->save();
                $timeslot = TimeSlot::findOrFail($appointment->time_slot_id);
                $timeslot->status = TimeSlot::BOOKED;
                $timeslot->save();
                $chat = Chat::findOrFail($service_message->chat_id);
                $chat->status = 1;
                $chat->save();
                $consultant = User::find($appointment->consultant_id);

                DB::commit();
                $title = "Booked an appointment! booking ID is #" . $appointment->id . ".";
                $body = "User ID: " . $user->id;
                $data = [
                    "user" => $user,  // Optionally, you can pass just the ID or a limited set of user info
                    "title" => $title,
                    "body" => $body,
                ];
                $admin = User::where('type', 'admin')->first();
                sendNotification($title, $body, $data, $user->id, $admin);
                sendNotification($user->first_name . " " . $user->last_name . " booked an appointment. Booking ID is #" . $appointment->id . ".", $body, $data, $admin->id, $consultant);
                sendNotification("You booked an appointment. Your booking ID is #" . $appointment->id . ".", $body, $data, $admin->id, $user);
                $token = $user->device_token;
                if ($token && $token != "undefined") {
                    $firebaseService = new FirebaseService();
                    $firebaseService->sendNotification($token, "Payment Successful", "Booking Payment Successful");
                }
                // Notification::route('firebase', $user->device_token)->notify(new PushNotification($title, $body));
                return view('front.payment-response', [
                    'status' => 'success',
                    'title' => 'Payment Successfull!',
                    'message' => 'Payment Successfull !',
                ]);
                // return redirect()->away('https://www.skyypllc.com/');

            }
            return view('front.payment-response', [
                'status' => 'danger',
                'title' => 'Error!',
                'message' => 'Something went wrong.'
            ]);
        }
    }

    public function cancel()
    {
        return view('front.payment-response', [
            'status' => 'danger',
            'title' => 'Payment Failed!',
            'message' => 'Something went wrong.'
        ]);
    }

}
