<?php

namespace App\Services;

use App\Models\Service;
use App\Models\ServiceConsultant;
use App\Models\User;
use App\Services\Common\BaseService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

/**
 * Class JsonResponseService
 */
class ServiceService extends BaseService
{
    public function index()
    {
        $service = Service::with('consultants', 'serviceCategory')
            ->when(request()->filled('status') && request('status') != 2, function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('title', 'like', '%'.request('search').'%');
                });
            })
            ->when(request()->filled('from'), function ($q) {
                $q->whereDate('created_at', '>=', request('from'))->whereDate('created_at', '<=', request('to'));
            })
            ->when(request()->filled('sort'), function ($q) {
                $q->where('status', '=', request('sort'));
            })
            ->when(request()->filled('service_category_id'), function ($q) {
                $q->where('service_category_id', '=', request('service_category_id'));
            })
            ->orderBy('index')
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);

        return $service;
    }

    public function activeServices()
    {
        $service = Service::where('status', 1)
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('title', 'like', '%'.request('search').'%');
                });
            })
            ->when(request()->filled('service_category_id'), function ($q) {
                $q->whereIn('service_category_id', '=', request('service_category_ids'));
            })
            ->with([
                'consultants' => function ($query) {
                    $query->where('status', 1); // Filter consultants with status = 1
                },
            ])
            ->orderBy('index')
            ->orderBy('id', 'desc')
            ->get();
        // ->paginate($this->pagination);

        return $service;
    }

    public function categoryServices($category_id)
    {
        $service = Service::where('status', 1)
            ->where('service_category_id', $category_id)
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('title', 'like', '%'.request('search').'%');
                });
            })
            ->when(request()->filled('service_category_id'), function ($q) {
                $q->whereIn('service_category_id', '=', request('service_category_ids'));
            })
            ->with([
                'consultants' => function ($query) {
                    $query->where('status', 1); // Filter consultants with status = 1
                },
            ])
            ->orderBy('index')
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);

        return $service;
    }

    public function consultantServices($user_id)
    {
        $services = User::findOrFail($user_id)
            ->services() // Get the services relationship
            ->where('title', 'like', '%'.request('search').'%') // Search by service title
            ->when(request()->filled('service_category_ids'), function ($q) {
                $q->whereHas('categories', function ($query) {
                    // Filter services by the categories provided in the request
                    $query->whereIn('categories.id', request('service_category_ids'));
                });
            })
            ->orderBy('index')
            ->orderBy('id', 'desc')
            ->get();

        return $services;
    }

    public function consultantServicesWithPagination($user_id)
    {
        $services = User::findOrFail($user_id)
            ->services() // Get the services relationship
            ->where('title', 'like', '%'.request('search').'%') // Search by service title
            ->when(request()->filled('service_category_ids'), function ($q) {
                $q->whereHas('categories', function ($query) {
                    // Filter services by the categories provided in the request
                    $query->whereIn('categories.id', request('service_category_ids'));
                });
            })
            ->orderBy('index')
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);

        return $services;
    }

    public function show($id)
    {
        $service = Service::with([
            'consultants' => function ($query) {
                $query->where('status', 1); // Filter consultants with status = 1
            },
        ])->findOrFail($id);

        return $service;
    }

    public function create($request)
    {
        DB::beginTransaction();
        $service = new Service;
        $service->title = $request->title;
        $service->description = $request->description;
        $service->charges = $request->charges;
        $service->status = $request->status;
        $service->service_category_id = $request->service_category_id;
        $service->save();
        if ($request->hasFile('images')) {
            storeArrayOfImage($request, 'images', Service::IMAGES_DIRECTORY, $service, 'images');
        }
        $consultant_ids = $request->input('consultant_ids'); // Array of user IDs (consultants)

        // Prepare the data for bulk insertion
        $consultants = [];
        foreach ($consultant_ids as $user_id) {
            $consultants[] = [
                'service_id' => $service->id,
                'user_id' => $user_id,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ];
        }
        ServiceConsultant::insert($consultants);
        DB::commit();

        return $service;
    }

    public function update($request, $id)
    {
        DB::beginTransaction();
        $service = Service::where('id', $id)->with('consultants')->firstOrFail(); // Use firstOrFail() for better error handling
        $service->title = $request->title;
        $service->description = $request->description;
        $service->charges = $request->charges;
        $service->status = $request->status;
        $service->service_category_id = $request->service_category_id;
        $service->save();
        // Remove old consultants
        if ($service->consultants()->exists()) {
            ServiceConsultant::where('service_id', $id)
                ->whereIn('user_id', $service->consultants->pluck('id'))
                ->delete();
        }
        // Handle the images if any
        if ($request->hasFile('images')) {
            storeArrayOfImage($request, 'images', Service::IMAGES_DIRECTORY, $service, 'images');
        }
        // Prepare the consultant data for bulk insertion
        $consultant_ids = $request->input('consultant_ids'); // Array of user IDs (consultants)
        if ($consultant_ids) {
            $consultants = [];
            foreach ($consultant_ids as $user_id) {
                $consultants[] = [
                    'service_id' => $service->id,
                    'user_id' => $user_id,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ];
            }
            ServiceConsultant::insert($consultants);
        }
        DB::commit();

        return $service;
    }

    public function accountStatus($userId)
    {
        $service = Service::find($userId);
        $service->status = ($service->status == 1) ? 0 : 1;
        $service->save();

        return $service;
    }

    public function delete($id)
    {
        $service = Service::findOrFail($id);
        $service->delete();

        return true;
    }

    public function reorder(array $servicesData)
    {
        DB::transaction(function () use ($servicesData) {
            foreach ($servicesData as $item) {
                if (! isset($item['id'], $item['index'])) {
                    continue;
                }

                Service::where('id', $item['id'])->update([
                    'index' => $item['index'],
                ]);
            }
        });

        return Service::orderBy('index')->orderBy('id', 'desc')->get();
    }
}
