<?php

namespace App\Services;

use App\Models\Plan;
use App\Services\Common\BaseService;
use Stripe\Stripe;
use Stripe\Plan as StripePlan;
use Stripe\Product;
use Stripe\Price;
/**
 * Class JsonResponseService
 * @package App\Services
 */
class PlanService extends BaseService
{
    public function __construct()
    {
        // Initialize Stripe with the secret key
        Stripe::setApiKey(config('stripe.secret_key'));
    }
    public function create($request)
    {
        $product = Product::create([
            'name' => $request->name,
        ]);
        // Create a price (plan) on Stripe
        $price = Price::create([
            'unit_amount' => $request->amount * 100,  // amount in cents
            // 'currency' => $request->currency,
            'currency' => 'usd',
            'recurring' => ['interval' => $request->interval], // 'month' or 'year'
            'product' => $product->id,
        ]);
        $plan = Plan::create([
            'stripe_product_id' => $product->id,
            'stripe_price_id' => $price->id,
            'name' => $request->name,
            'description' => $request->description,
            'amount' => $request->amount,
            // 'currency' => $request->currency,
            'currency' => 'usd',
            'interval' => $request->interval,
            'status' => 1,
        ]);
        return $plan;
    }

    public function update($id, $request)
    {
        $plan = Plan::findOrFail($id);
        // Retrieve the existing Stripe price and product
        $stripePrice = Price::retrieve($plan->stripe_price_id);
        $stripeProduct = Product::retrieve($plan->stripe_product_id);

        // Update the product and price information if provided
        if (isset($request->name) && $request->name != $plan->name) {
            $stripeProduct->name = $request->name;
            $plan->name = $request->name;
            $stripeProduct->save();
        }
        if (isset($request->description) && $request->description != $plan->description) {
            $stripeProduct->description = $request->description;
            $plan->description = $request->description;
            $stripeProduct->save();
        }
        if (isset($request->amount) && $request->amount != $plan->amount) {
            $stripePrice->unit_amount = $request->amount * 100; // Update amount in cents
            $plan->amount = $request->amount; // Update amount in cents
            $stripePrice->save();
        }

        if (isset($request->interval) && $request->interval != $plan->interval) {
            // Stripe does not allow updating the recurring interval directly
            // You must create a new Price (if changing interval) and update the local database accordingly
            $newPrice = Price::create([
                'unit_amount' => $request->amount * 100, // New amount
                // 'currency' => $request->currency,
                'currency' => 'usd',
                'recurring' => ['interval' => $request->interval], // New interval
                'product' => $stripeProduct->id,
            ]);

            // Update the plan with the new price ID and interval in the local database
            $plan->update([
                'stripe_price_id' => $newPrice->id,
                'interval' => $request->interval,
            ]);

            // Optionally, delete the old price if it’s no longer needed
            $stripePrice->delete();
        }
        $plan->save();
        // Update the local database record with the new data
        return $plan;
    }

    public function delete($id)
    {
        $plan = Plan::findOrFail($id);

        // Delete the Stripe price and product associated with the plan
        $stripePrice = Price::retrieve($plan->stripe_price_id);
        $stripeProduct = Product::retrieve($plan->stripe_product_id);

        // Delete the price and product on Stripe
        $stripePrice->delete();
        $stripeProduct->delete();

        // Delete the local database record
        $plan->delete();

        return 'Plan deleted successfully';
    }

    public function status($id)
    {
        $plan = Plan::findOrFail($id);
        $plan->status = ($plan->status == 1) ? 0 : 1;
        $plan->save();
        return $plan;
    }

    public function list()
    {
        $plans = Plan::when(request()->filled('status') && request('status') != 'all', function ($q) {
            $q->whereStatus(request('status'));
        })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('from'), function ($q) {
                $q->whereDate('created_at', '>=', request('from'))
                    ->whereDate('created_at', '<=', request('to'));
            })
            ->orderBy('id', 'desc') // Then order by id in descending order
            ->paginate($this->pagination);

        return $plans;
    }

    public function listActive()
    {
        $plans = Plan::where('status', 1)->when(request()->filled('status') && request('status') != 'all', function ($q) {
            $q->whereStatus(request('status'));
        })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('id', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('from'), function ($q) {
                $q->whereDate('created_at', '>=', request('from'))
                    ->whereDate('created_at', '<=', request('to'));
            })
            ->orderBy('id', 'desc') // Then order by id in descending order
            ->paginate($this->pagination);

        return $plans;
    }
}
