<?php


namespace App\Services\Common;

use App\Http\Resources\Admin\UserListingResource;
use App\Http\Resources\User\ConsultantResource;
use App\Http\Resources\User\UserProfileWithAllDataResource;
use App\Models\Image;
use App\Services\StripeService;
use Carbon\Carbon;
use Exception;
use Hash;
use DB;
use App\Models\User;
use HelperConstants;
use App\Models\PasswordReset;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Validation\ValidationException;
use App\Notifications\{SignupActivation, PasswordResetRequest};
use App\Notifications\SendAdminNotification;


class UserService extends CrudService
{

    // use UploadAble;


    public function create(object $request, $type = 'user')
    {
        DB::beginTransaction();
        $stripeService = new StripeService();
        $customer = $stripeService->createCustomer($request->first_name . " " . $request->last_name, $request->email);
        $user = new User();
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->stripe_customer_id = $customer->id;
        $user->email = $request->email;
        $user->type = $type;
        $user->email_verified_at = Carbon::now();
        $user->password = Hash::make($request->password);
        $user->dial_code = $request->dial_code;
        $user->phone = $request->phone;
        $user->country_code = $request->country_code;
        $user->device_token = $request->device_token;
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $image = $saveFile['fileName'];
            $user->image = $image;
        }
        $user->save();

        // Code to send notification to admin for approve this user
        // $title = $user->first_name . " " . $user->last_name . " just created a new account.";
        // $body = $user->id;
        // $data = [
        //     "title" => $title,
        //     "body" => $user->id,
        // ];
        // $admin = User::where('type', 'admin')->first();
        // sendNotification($title, $body, $data, $user->id, $admin);
        DB::commit();
        return $user;
    }
    public function createConsultant(object $request)
    {
        DB::beginTransaction();
        $stripeService = new StripeService();
        $customer = $stripeService->createCustomer($request->first_name . " " . $request->last_name, $request->email);
        $user = new User();
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->stripe_customer_id = $customer->id;
        $user->email = $request->email;
        $user->type = "consultant";
        $user->email_verified_at = Carbon::now();
        $user->password = Hash::make($request->password);
        $user->dial_code = $request->dial_code;
        $user->phone = $request->phone;
        $user->country_code = $request->country_code;
        $user->device_token = $request->device_token;
        $user->personal_details = $request->personal_details;
        $user->licence_no = $request->licence_no;
        $user->license_expiration = $request->license_expiration;
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $image = $saveFile['fileName'];
            $user->image = $image;
        }
        if ($request->has('cover_photo')) {
            $saveFile = saveFile($request->cover_photo, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $cover_photo = $saveFile['fileName'];
            $user->cover_photo = $cover_photo;
        }
        $user->save();
        // $user = $this->updateImage($request, $user->id);
        if ($request->hasFile('license_images')) {
            storeArrayOfImage($request, 'license_images', User::LICENSE_IMAGES_DIRECTORY, $user, "images", 'license_images');
        }
        if ($request->hasFile('attachments')) {
            storeArrayOfImage($request, 'attachments', User::LICENSE_ATTACHMENTS_DIRECTORY, $user, "images", 'attachments');
        }

        // Code to send notification to admin for approve this user
        // $title = $user->first_name . " " . $user->last_name . " just created a new account.";
        // $body = $user->id;
        // $data = [
        //     "title" => $title,
        //     "body" => $user->id,
        // ];
        // $admin = User::where('type', 'admin')->first();
        // sendNotification($title, $body, $data, $user->id, $admin);
        DB::commit();
        return $user;
    }

    public function updateConsultant(object $request, $id)
    {
        DB::beginTransaction();
        $user = User::findOrFail($id); // Assuming consultant already exists, use the passed ID

        if ($request->has('first_name')) {
            $user->first_name = $request->first_name;
        }

        if ($request->has('last_name')) {
            $user->last_name = $request->last_name;
        }

        if ($request->has('dial_code')) {
            $user->dial_code = $request->dial_code;
        }

        if ($request->has('phone')) {
            $user->phone = $request->phone;
        }

        if ($request->has('country_code')) {
            $user->country_code = $request->country_code;
        }

        if ($request->has('device_token')) {
            $user->device_token = $request->device_token;
        }

        if ($request->has('personal_details')) {
            $user->personal_details = $request->personal_details;
        }

        if ($request->has('licence_no')) {
            $user->licence_no = $request->licence_no;
        }

        if ($request->has('license_expiration')) {
            $user->license_expiration = $request->license_expiration;
        }

        if ($request->hasFile('image')) {
            deleteFile(HelperConstants::PROFILE_IMAGE_DIRECTORY . "/" . $user->image);
            $saveFile = saveFile($request->image, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $user->image = $saveFile['fileName'];
        }

        if ($request->hasFile('cover_photo')) {
            deleteFile(HelperConstants::PROFILE_IMAGE_DIRECTORY . "/" . $user->cover_photo);
            $saveFile = saveFile($request->cover_photo, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $user->cover_photo = $saveFile['fileName'];
        }
        $user->save();

        if ($request->hasFile('license_images')) {
            storeArrayOfImage($request, 'license_images', User::LICENSE_IMAGES_DIRECTORY, $user, "images", 'license_images');
        }
        if ($request->hasFile('attachments')) {
            storeArrayOfImage($request, 'attachments', User::LICENSE_ATTACHMENTS_DIRECTORY, $user, "images", 'attachments');
        }
        if ($request->has('delete_files')) {
            foreach ($request->delete_files as $value) {
                $image = Image::find($value);
                if ($image) {
                    deleteImage($image);
                }
            }
        }
        DB::commit();
        return $user;
    }

    public function updateImage($request, $id)
    {
        $user = User::find($id);
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $image = $saveFile['fileName'];
            $user->image = $image;
        }
        if ($request->has('cover_photo')) {
            $saveFile = saveFile($request->cover_photo, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $image = $saveFile['fileName'];
            $user->cover_photo = $image;
        }
        $user->save();
        return $user;
    }
    public function checkPassword($request): bool
    {
        return !Hash::check($request->password, auth()->user()->password);
    }

    public function updateFaceId($request): bool
    {
        auth()->user()->is_biometric_enable = 1;
        auth()->user()->face_id = $request->public_key;
        auth()->user()->save();
        return true;
    }

    public function uploadImages($request)
    {
        $user = User::find(auth()->user()->id);
        storeArrayOfImage($request, "images", User::IMAGES_DIRECTORY, $user, "images");
        return $user;
    }


    public function deleteImage($image)
    {
        deleteImage($image);
        return auth()->user();
    }


    public function uploadSocialLinks($request)
    {
        $user = auth()->user();

        // Separate data for updating and inserting
        $updateData = [];
        $insertData = [];
        $typesInRequest = [];

        // Collect types from the request to filter existing links
        foreach ($request->social_links as $socialLink) {
            $typesInRequest[] = $socialLink['type'];
        }

        // Get existing social links for the types provided in the request
        $existingLinks = $user->socialLinks()->whereIn('type', $typesInRequest)->get()->keyBy('type');

        foreach ($request->social_links as $socialLink) {
            if ($existingLinks->has($socialLink['type'])) {
                // Prepare data for updating
                $updateData[] = [
                    'id' => $existingLinks[$socialLink['type']]->id,
                    'link' => $socialLink['link']
                ];
            } else {
                // Prepare data for inserting
                $insertData[] = [
                    'link' => $socialLink['link'],
                    'type' => $socialLink['type'],
                    'user_id' => $user->id,
                    'created_at' => now(),
                    'updated_at' => now()
                ];
            }
        }

        // Perform batch update for existing links
        if (!empty($updateData)) {
            foreach ($updateData as $data) {
                \DB::table('user_social_links')
                    ->where('id', $data['id'])
                    ->update(['link' => $data['link'], 'updated_at' => now()]);
            }
        }

        // Perform batch insert for new links
        if (!empty($insertData)) {
            \DB::table('user_social_links')->insert($insertData);
        }
        return $user;
    }
    public function disableFaceId(): bool
    {
        auth()->user()->is_biometric_enable = 0;
        auth()->user()->face_id = "";
        auth()->user()->save();
        return true;
    }

    public function enable_location($userId)
    {
        $user = User::find($userId);
        $user->is_location_enable = ($user->is_location_enable == 1) ? 0 : 1;
        $user->save();
        return $user;
    }

    public function changeStatusOfAProperty($userId, $property)
    {
        $user = User::find($userId);
        $user->$property = ($user->$property == 1) ? 0 : 1;
        $user->save();
        return $user;
    }



    public function update($request)
    {
        auth()->user()->name = $request->name;
        auth()->user()->dialing_code = $request->dialing_code;
        auth()->user()->phone = $request->phone;
        auth()->user()->dob = $request->dob;
        auth()->user()->country = $request->country;

        if ($request->has('image')) {
            $saveFile = saveFile($request->image, HelperConstants::PROFILE_image_DIRECTORY, HelperConstants::UPLOAD_DISK);
            auth()->user()->image = $saveFile['fileName'];
        }

        auth()->user()->save();

        return auth()->user();
    }

    public function getAll()
    {
        return User::all();
    }

    public function show($id)
    {
        return User::find($id);
    }

    public function updatePassword($request, $id)
    {
        $user = User::find($id);

        if ($request->has('password')) {
            $user->password = Hash::make($request->password);
        }
        $user->save();

        return $user;
    }

    public function updateStatus($request, $id)
    {
        $user = User::find($id);
        $user->status = $request->status;
        $user->save();

        return $user;
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        User::findOrFail($id)->delete();
        DB::commit();
        return response()->json(['status' => true, 'message' => 'deleted successfully']);
    }

    public function getUserByEmail($email)
    {
        return User::where('email', $email)->first();
    }

    public function getUserByPhone($phone)
    {
        return User::where('phone', $phone)->first();
    }

    public function getUserByFaceId($face_id)
    {
        return User::where('face_id', $face_id)->first();
    }

    public function checkLogin($request, $type = 'user')
    {
        if ($request->exists('face_id') && $request->filled('phone')) {
            $user = $this->getUserByFaceId($request->face_id);

            if (!$user) {
                throw ValidationException::withMessages([
                    'phone' => ['Incorrect face Id'],
                ]);
            }
        } else {
            $user = $this->getUserByEmail($request->email);
            if (!$user || !Hash::check($request->password, $user->password)) {
                throw new AuthenticationException(
                    "The provided credentials are incorrect."
                );
            }
        }
        if ($user->type !== $type) {
            throw new AuthenticationException(
                "You're not valid user to access this route"
            );
        }
        if ($user->email_verified_at == null) {
            throw new AuthenticationException(
                "Your account is not verified"
            );
        }
        if ($user->status == 0) {
            throw new AuthenticationException(
                "Your account has been deactivated"
            );
        }
        if ($request->latitude && $request->longitude) {
            $this->updateDeviceLocationByEmail($user->email, $request->latitude, $request->longitude);
        }
        $user = User::find($user->id);
        $this->updateDeviceToken($user->email, $request->device_token);
        $resUser = "";
        if ($user->type == "consultant") {
            $resUser = new ConsultantResource($user);
        } else {
            $resUser = new UserListingResource($user);
        }
        return ['token' => $user->createToken('123', ['account:verified'])->plainTextToken, 'user' => $resUser];
    }

    public function checkBiometricLogin($request, $type = 'user')
    {
        $email = $request->email;
        $payload = $request->input('payload');
        $signature = $request->input('signature');

        // Fetch the public key associated with the external identifier
        $publicKey = DB::table('users')->where('email', $email)->value('face_id');

        if (!$publicKey) {
            $this->removePublicKey($request->email);
            throw new AuthenticationException(
                "Public key not found"
            );
        }
        $formattedPublicKey = "-----BEGIN PUBLIC KEY-----\n" . wordwrap($publicKey, 64, "\n", true) . "\n-----END PUBLIC KEY-----";
        $isValid = openssl_verify($payload, base64_decode($signature), $formattedPublicKey, OPENSSL_ALGO_SHA256);

        if ($isValid) {
            $user = $this->getUserByEmail($request->email);
            if ($user->type !== $type) {
                throw new AuthenticationException(
                    "You're not valid user to access this route"
                );
            }
            if ($user->email_verified_at == null) {
                throw new AuthenticationException(
                    "Your account is not verified"
                );
            }
            if ($user->status == 0) {
                throw new AuthenticationException(
                    "Your account has been deactivated"
                );
            }
            $this->updateDeviceToken($user->email, $request->device_token);
            if ($request->latitude && $request->longitude) {
                $this->updateDeviceLocationByEmail($user->email, $request->latitude, $request->longitude);
            }
            return ['token' => $user->createToken('123', ['account:verified'])->plainTextToken, 'user' => new UserListingResource($user)];
        } else {
            $this->removePublicKey($request->email);
            throw new AuthenticationException(
                "Invalid payload"
            );
        }
    }
    public function removePublicKey($email)
    {
        $user = User::where('email', $email)->first();
        if ($user) {
            $user->is_biometric_enable = 0;
            $user->face_id = '';
            $user->save();
        }
    }
    public function updateDeviceToken($email, $deviceToken)
    {
        $user = $this->getUserByEmail($email);
        $user->device_token = $deviceToken;
        $user->save();
    }

    public function updateDeviceTokenByPhone($phone, $deviceToken)
    {
        $user = $this->getUserByPhone($phone);
        $user->device_token = $deviceToken;
        $user->save();
    }


    public function updateDeviceLocationByEmail($email, $latitude, $longitude)
    {
        $user = $this->getUserByEmail($email);
        $user->longitude = $longitude;
        $user->latitude = $latitude;
        $user->save();
    }

    public function reset($request)
    {
        $user = $this->getUserByEmail($request->email);

        if (!$user) {
            throw ValidationException::withMessages([
                'email' => ['Email is not registered'],
            ]);
        }

        $passwordReset = $this->generateToken($user->email);

        if ($user && $passwordReset)
            $user->notify(
                new PasswordResetRequest($user, $passwordReset->token)
            );
        return true;

    }

    public function verifyAndUpdatePassword($request)
    {
        $user = $this->getUserByEmail($request->email);
        return $this->updatePassword($request, $user->id);
    }

    public function sendAccountVerification($user)
    {
        $verification = $this->generateToken($user->email);
        if ($user && $verification)
            $user->notify(
                new SignupActivation($user, $verification->token)
            );
    }

    public function generateToken($email)
    {
        return PasswordReset::updateOrCreate(
            ['email' => $email],
            [
                'email' => $email,
                'token' => strtolower(rand(1000, 9999)),
                'created_at' => now()
            ]
        );
    }

    public function markVerify($request, $email)
    {
        $code = $this->getToken($request);

        if (!$code || $code->email != $email) {
            return false;
        }
        $user = $this->getUserByEmail($email);

        $user->verified = 1;

        // Remove after development of admin
        $user->status = 1;

        $user->save();

        $this->deleteToken($request->token);

        return true;
    }

    public function getToken($request)
    {
        return PasswordReset::where('token', $request->token)->first();
    }

    public function deleteToken($token)
    {
        PasswordReset::where('token', $token)->delete();
    }

    public function setLocation($request, $userId)
    {
        $user = User::where('id', $userId)->first();

        if ($user->location) {

            $location['latitude'] = $request->latitude;
            $location['longitude'] = $request->longitude;
            $location['text'] = $request->location;
            $user->location()->update($location);
        } else {

            $location = new \App\Models\Location;
            $location->latitude = $request->latitude;
            $location->longitude = $request->longitude;
            $location->text = $request->location;
            $user->location()->save($location);
        }

        return $user;
    }

    public function pushNotificationStatus($userId, $status)
    {
        $user = User::find($userId);
        $user->push_notification = $status;
        $user->save();

        return $user;
    }

    public function logout($request)
    {
        return $request->user()->currentAccessToken()->delete();
    }

    public function listing($request, $pagination)
    {
        return User
            ->whereNotIn('status', [User::PENDING_STATUS, User::REJECT_STATUS])
            ->when(request()->filled('status'), function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            })
            ->when(request()->filled('start'), function ($q) {
                $q->whereDate('created_at', '>=', request('start'))->whereDate('created_at', '<=', request('end'));
            })
            ->orderBy('id', 'DESC')
            ->paginate($pagination);
    }

    public function requestListing($request, $pagination)
    {
        return User::
            where(function ($q) {
                $q->where('status', User::PENDING_STATUS)->orWhere('status', User::REJECT_STATUS);
            })
            ->when(request()->filled('status'), function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            })
            ->when(request()->filled('start'), function ($q) {
                $q->whereDate('created_at', '>=', request('start'))->whereDate('created_at', '<=', request('end'));
            })
            ->orderBy('id', 'DESC')
            ->paginate($pagination);
    }

    public function accountStatus($userId, $status)
    {
        $user = User::find($userId);
        $user->status = $status;
        $user->save();

        return $user;
    }
}
