<?php

namespace App\Services;

use App\Models\Appointment;
use App\Models\ConsultantInventory;
use App\Models\Image;
use App\Models\SubCategory;
use App\Models\TimeSlot;
use App\Models\User;
use App\Services\Common\BaseService;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

/**
 * Class JsonResponseService
 * @package App\Services
 */
class AppointmentService extends BaseService
{
    public function index()
    {
        $appointments = Appointment::where('user_id', auth()->user()->id)
            ->when(request()->filled('status') && request('status') != 'all', function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('id', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('from'), function ($q) {
                $q->whereDate('created_at', '>=', request('from'))
                    ->whereDate('created_at', '<=', request('to'));
            })
            // Custom ordering to show "upcoming" first
            ->orderByRaw("FIELD(status, 'upcoming') DESC") // 'upcoming' first
            ->orderBy('status', 'desc') // Then order by status in descending order
            ->paginate($this->pagination);
        return $appointments;
    }

    public function userAppointment($userId)
    {
        $appointments = Appointment::where('user_id', $userId)
            ->when(request()->filled('status') && request('status') != 'all', function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('id', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('fromDate') && request()->filled('toDate'), function ($q) {
                $q->whereDate('booking_date', '>=', request('fromDate'))
                    ->whereDate('booking_date', '<=', request('toDate'));
            })
            ->when(request()->filled('appointmentFromDate') && request()->filled('appointmentToDate'), function ($q) {
                // $q->whereDate('booking_date', '>=', request('fromDate'))
                //     ->whereDate('booking_date', '<=', request('toDate'));
            })
            // Custoem ordering to show "upcoming" first
            ->orderByRaw("FIELD(status, 'upcoming') DESC") // 'upcoming' first
            ->orderBy('status', 'desc') // Then order by status in descending order
            ->paginate($this->pagination);
        return $appointments;
    }
    public function userPaidAppointment($userId)
    {
        $appointments = Appointment::where('user_id', $userId)->where('is_paid', 1)
            ->when(request()->filled('status') && request('status') != 'all', function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('id', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('fromDate') && request()->filled('toDate'), function ($q) {
                $q->whereDate('booking_date', '>=', request('fromDate'))
                    ->whereDate('booking_date', '<=', request('toDate'));
            })
            ->when(request()->filled('appointmentFromDate') && request()->filled('appointmentToDate'), function ($q) {
                // $q->whereDate('booking_date', '>=', request('fromDate'))
                //     ->whereDate('booking_date', '<=', request('toDate'));
            })
            // Custoem ordering to show "upcoming" first
            ->orderByRaw("FIELD(status, 'upcoming') DESC") // 'upcoming' first
            ->orderBy('status', 'desc') // Then order by status in descending order
            ->paginate($this->pagination);
        return $appointments;
    }
    public function consultantAppointment($consultantId)
    {
        $appointments = Appointment::where('consultant_id', $consultantId)
            ->when(request()->filled('status') && request('status') != 'all', function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('id', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('fromDate') && request()->filled('toDate'), function ($q) {
                $q->whereDate('booking_date', '>=', request('fromDate'))
                    ->whereDate('booking_date', '<=', request('toDate'));
            })
            ->when(request()->filled('appointmentFromDate') && request()->filled('appointmentToDate'), function ($q) {
                // $q->whereDate('booking_date', '>=', request('fromDate'))
                //     ->whereDate('booking_date', '<=', request('toDate'));
            })
            // Custoem ordering to show "upcoming" first
            ->orderByRaw("FIELD(status, 'upcoming') DESC") // 'upcoming' first
            ->orderBy('status', 'desc') // Then order by status in descending order
            ->paginate($this->pagination);
        return $appointments;
    }

    public function adminAppointments()
    {
        $appointments = Appointment::when(request()->filled('status') && request('status') != 'all', function ($q) {
            $q->whereStatus(request('status'));
        })
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('id', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('from'), function ($q) {
                $q->whereDate('created_at', '>=', request('from'))
                    ->whereDate('created_at', '<=', request('to'));
            })
            // Custom ordering to show "upcoming" first
            ->orderByRaw("FIELD(status, 'upcoming') DESC") // 'upcoming' first
            ->orderBy('status', 'desc') // Then order by status in descending order
            ->paginate($this->pagination);
        return $appointments;
    }

    public function checkAppointment($user_id, $timeslot)
    {
        $date = $timeslot->date;
        $time = $timeslot->time;
        $appointment = Appointment::where('user_id', $user_id)
            ->whereIn('status', [Appointment::UPCOMING, Appointment::INPROGRESS])
            ->whereHas('timeslot', function ($q) use ($date, $time) {
                $q->whereDate('date', $date)
                    ->whereTime('time', $time);
            })
            ->first();
        return $appointment;
    }

    public function view($id)
    {
        $appointment = Appointment::where('id', $id)
            ->with(['consultant', 'user', 'service', 'timeslot', 'inventories'])
            ->first();

        if ($appointment && $appointment->timeslot) {
            $start = Carbon::parse($appointment->timeslot->time)->format('h:i A');
            $end = Carbon::parse($appointment->timeslot->time)->addHour()->format('h:i A');
            $appointment->timeslot->time_range = "$start - $end";
        }

        return $appointment;
    }

    public function create($request)
    {
        $appointment = new Appointment();
        $appointment->first_name = $request->first_name;
        $appointment->last_name = $request->last_name;
        $appointment->email = $request->email;
        $appointment->phone = $request->phone;
        $appointment->country_code = $request->country_code;
        $appointment->dial_code = $request->dial_code;
        $appointment->booking_date = Carbon::now()->toDateString();
        $appointment->status = Appointment::UPCOMING;
        $appointment->service_id = $request->service_id;
        $appointment->time_slot_id = $request->time_slot_id;
        $appointment->consultant_id = $request->consultant_id;
        $appointment->user_id = auth()->user()->id;
        $appointment->save();
        return Appointment::where('id', $appointment->id)->with('service')->first();
    }

    public function update($request, $id)
    {
        $appointment = Appointment::findOrFail($id);

        // Update report and status
        if ($request->has('report')) {
            $appointment->report = $request->report;
            $appointment->status = Appointment::PAST;
        }

        if ($request->has('form_json')) {
            $appointment->form_json = $request->form_json;
        }

        if ($request->has('chart_id')) {
            $appointment->chart_id = $request->chart_id;
        }

        if ($request->has('chart_json')) {
            $appointment->chart_json = $request->chart_json;
        }

        $appointment->save();

        // Handle inventory update
        if ($request->has('inventory') && is_array($request->inventory)) {
            $appointmentId = $id;
            foreach ($request->inventory as $item) {
                // Subtract quantity from SubCategory
                $subCategory = SubCategory::find($item['sub_category_id']);
                if ($subCategory) {
                    // Optional: Validate sufficient stock before subtracting
                    if ($subCategory->quantity < $item['quantity']) {
                        throw new \Exception("Not enough stock in sub-category: {$subCategory->name}");
                    }

                    $subCategory->quantity = max(0, $subCategory->quantity - $item['quantity']);
                    $subCategory->save();
                }

                // Create ConsultantInventory record
                ConsultantInventory::create([
                    'user_id' => auth()->id(),
                    'appointment_id' => $appointmentId,
                    'category_id' => $item['category_id'],
                    'sub_category_id' => $item['sub_category_id'],
                    'quantity' => $item['quantity'],
                ]);
            }
        }

        // Upload new documents
        if ($request->hasFile('documents')) {
            storeArrayOfImage($request, 'documents', Appointment::FILES_DIRECTORY, $appointment, "documents");
        }

        // Delete selected documents
        if ($request->has('delete_documents')) {
            foreach ($request->delete_documents as $value) {
                $image = Image::find($value);
                if ($image) {
                    deleteImage($image);
                }
            }
        }

        // Send notification to consultant
        $title = "Your appointment ID " . $appointment->id . " report has been updated!";
        $body = "Consultant ID: " . $appointment->consultant_id;
        $data = [
            "user" => $appointment->consultant_id,
            "title" => $title,
            "body" => $body,
        ];

        $user = User::find($appointment->user_id);
        sendNotification($title, $body, $data, $appointment->consultant_id, $user);

        $token = $user->device_token;
        if ($token && $token !== "undefined") {
            $firebaseService = new FirebaseService();
            $firebaseService->sendNotification($token, "Appointment Report", $title);
        }



        return Appointment::where('id', $appointment->id)->with('service', 'documents')->first();
    }

    public function createWeb($request, $user)
    {
        $appointment = new Appointment();
        $appointment->first_name = $request->first_name;
        $appointment->last_name = $request->last_name;
        $appointment->email = $request->email;
        $appointment->phone = $request->phone;
        $appointment->country_code = $request->country_code;
        $appointment->dial_code = $request->dial_code;
        $appointment->booking_date = Carbon::now()->toDateString();
        $appointment->status = Appointment::UPCOMING;
        $appointment->service_id = $request->service_id;
        $appointment->time_slot_id = $request->time_slot_id;
        $appointment->consultant_id = $request->consultant_id;
        $appointment->user_id = $user->id;
        $appointment->save();
        return Appointment::where('id', $appointment->id)->with('service')->first();
    }
}
