<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class OrderNotification extends Notification
{
    use Queueable;

    protected $order;

    /**
     * Create a new notification instance.
     */
    public function __construct($order)
    {
        $this->order = $order;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        // Extract contact information (assuming 'type' = 'contact')
        $contactAddress = collect($this->order['address'])->firstWhere('type', 'contact');

        // Get user details from the contact address (or fallback if not available)
        $userName = $contactAddress['first_name'] . ' ' . $contactAddress['last_name'];
        $userEmail = $contactAddress['email'] ?: 'No email provided';
        $userPhone = $contactAddress['phone'];

        // Prepare the order details (product names, quantities, prices)
        $orderDetails = '';
        foreach ($this->order['details'] as $item) {
            $orderDetails .= $item['product']['name'] . ' - Quantity: ' . $item['quantity'] . ' - Price: $' . $item['price'] . "\n";
        }

        // Total price and other info
        $totalPrice = $this->order['total_price'];
        $totalQuantity = $this->order['total_quantity'];
        $status = ucfirst($this->order['status']);
        $rejectionReason = $this->order['rejection_reason'] ?? 'N/A';

        // Compose the email
        return (new MailMessage)
            ->subject('Order Details - Order ID: ' . $this->order['id'])
            ->line('Hello ' . $userName . ',')
            ->line('Thank you for your order. Here are the details of your order:')
            ->line('Order ID: ' . $this->order['id'])
            ->line('Total Quantity: ' . $totalQuantity)
            ->line('Total Price: $' . $totalPrice)
            ->line('Order Status: ' . $status)
            ->line('Rejection Reason (if applicable): ' . $rejectionReason)
            ->line('Shipping and Billing Address:')
            ->line('Shipping Address: ' . $this->order['address'][0]['address'] . ', ' . $this->order['address'][0]['city'] . ', ' . $this->order['address'][0]['state'] . ', ' . $this->order['address'][0]['zip_code'])
            ->line('Billing Address: ' . $this->order['address'][1]['address'] . ', ' . $this->order['address'][1]['city'] . ', ' . $this->order['address'][1]['state'] . ', ' . $this->order['address'][1]['zip_code'])
            ->line('Contact Info: Email - ' . $userEmail . ', Phone - ' . $userPhone)
            ->line('Order Items:')
            ->line($orderDetails)
            ->line('Thank you for using our application!');
            // ->action('View Order', url('/order/' . $this->order['id']));
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            // Additional details can be added here
        ];
    }
}
