<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $guarded = ['id'];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $dates = ['deleted_at'];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    public function images()
    {
        return $this->morphMany(Image::class, 'imageable');
    }

    public function licenseImages()
    {
        return $this->morphMany(Image::class, 'imageable')->where('category', 'license_images');
    }

    public function attachments()
    {
        return $this->morphMany(Image::class, 'imageable')->where('category', 'attachments');
    }

    public function ratings()
    {
        return $this->hasMany(Rating::class, 'consultant_id'); // Ratings are linked by 'consultant_id'
    }

    // Define the accessor for 'image_url'
    public function getImageUrlAttribute()
    {
        return $this->image ? asset("/images/profile/img/{$this->image}") : null;
    }

    public function getCoverPhotoUrlAttribute()
    {
        return $this->image ? asset("/images/profile/img/{$this->cover_photo}") : null;
    }

    public function getFullNameAttribute()
    {
        return $this->first_name.' '.$this->last_name;
    }

    public const IMAGES_DIRECTORY = 'images/profile/img/';

    public const LICENSE_IMAGES_DIRECTORY = 'images/user/license/';

    public const LICENSE_ATTACHMENTS_DIRECTORY = 'images/user/attachments/';

    public function feedbacks()
    {
        return $this->hasMany(Feedback::class);
    }

    public function timeSlots()
    {
        return $this->hasMany(TimeSlot::class, 'consultant_id');
    }

    public function services(): BelongsToMany
    {
        return $this->belongsToMany(Service::class, 'service_consultants', 'user_id', 'service_id');
    }
}
