<?php

namespace App\Http\Requests\TimeSlot;

use Illuminate\Foundation\Http\FormRequest;
use Carbon\Carbon;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;

class CreateTimeSlotRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true; // If no authorization logic is needed
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'consultant_id' => 'required|exists:users,id', // Ensure consultant exists
            'timeslots' => 'required|array',
            'timeslots.*.day' => 'required|string|in:Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday', // Validate valid day names
            'timeslots.*.is_enabled' => 'required',

            // Conditional validation for slots based on is_enabled
            'timeslots.*.slots' => 'array', // slots should be an array, no minimum defined yet

            // Updated rules with condition on is_enabled for time and status validation
            'timeslots.*.slots.*.time' => [
                'required_if:timeslots.*.is_enabled,true',
                'date_format:H:i',
                function ($attribute, $value, $fail) {
                    $isEnabled = filter_var(request()->input('timeslots.*.is_enabled'), FILTER_VALIDATE_BOOLEAN);
                    if ($isEnabled && empty($value)) {
                        $fail("Time is required for the slot when is_enabled is true.");
                    }
                }
            ],

            'timeslots.*.slots.*.status' => [
                'required_if:timeslots.*.is_enabled,true',
                'string',
                'in:available,cancelled,booked',
                function ($attribute, $value, $fail) {
                    $isEnabled = filter_var(request()->input('timeslots.*.is_enabled'), FILTER_VALIDATE_BOOLEAN);
                    if ($isEnabled && empty($value)) {
                        $fail("Status is required for the slot when is_enabled is true.");
                    }
                }
            ],

            // Custom validation to ensure at least one slot is provided if is_enabled is true
            'timeslots.*' => function ($attribute, $value, $fail) {
                $isEnabled = filter_var($value['is_enabled'], FILTER_VALIDATE_BOOLEAN); // Cast to boolean
                if ($isEnabled && empty($value['slots'])) {
                    $fail("At least one slot must be provided for the day: {$value['day']} when is_enabled is true.");
                }
            },
        ];
    }

    /**
     * Get the custom messages for validation errors.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'consultant_id.required' => 'Consultant ID is required.',
            'consultant_id.exists' => 'The specified consultant does not exist.',
            'timeslots.required' => 'Timeslot information is required.',
            'timeslots.*.day.in' => 'The day must be one of: Monday, Tuesday, Wednesday, Thursday, Friday, Saturday, Sunday.',
            'timeslots.*.slots.*.time.required_if' => 'At least one slot must be provided for each enabled day.',
            'timeslots.*.slots.*.status.in' => 'The status must be either available, cancelled, or booked.',
            'timeslots.*.slots.*.time.date_format' => 'Time must be in the format HH:mm.',
        ];
    }
}
