<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\MainApiController;
use App\Http\Requests\Password\ChangePasswordRequest;
use App\Http\Requests\Password\ResetPassword;
use App\Http\Requests\User\LoginRequest;
use App\Http\Requests\User\UserProfileUpdateRequest;
use App\Http\Requests\User\UserRegisterRequest;
use App\Http\Requests\User\UserSetPasswordRequest;
use App\Http\Requests\VerifyActivationCodeRequest;
use App\Http\Resources\Admin\UserListingResource;
use App\Http\Resources\DefaultResource;
use App\Models\Order;
use App\Models\User;
use App\Services\Common\UserService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AuthController extends MainApiController
{
    private $userService;

    public function __construct(UserService $userService)
    {
        parent::__construct();
        $this->userService = $userService;
    }

    public function dashboardStats(Request $request)
    {
        $monthlyTrend = User::selectRaw('COUNT(id) as count')->whereRaw("created_at LIKE '".date('Y-m')."%'")->where('type', 'user')->first();
        $totalUsers = User::where('type', 'user')->count();
        $monthlyTrend = (isset($monthlyTrend->count) && $monthlyTrend->count > 0 ? round((($monthlyTrend->count / $totalUsers) * 100), 2) : 0);
        $monthlyTrendApplication = User::selectRaw('COUNT(id) as count')->whereRaw("created_at LIKE '".date('Y-m')."%'")->first();
        $totalApplications = User::count();
        $monthlyTrendApplication = (isset($monthlyTrendApplication->count) && $monthlyTrendApplication->count > 0 ? round((($monthlyTrendApplication->count / $totalApplications) * 100), 2) : 0);
        $currentYear = request('year');
        // Prepare an array to hold the results
        $monthsCountUsers = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month
        $monthsCountEarnings = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month

        // Loop through each month from January to the current month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Count the number of users created in this month
            $count = User::where('type', 'user')->whereBetween('created_at', [$startOfMonth, $endOfMonth])->count();

            // Store the count in the corresponding month index (month - 1 because array index starts at 0)
            $monthsCountUsers[$month - 1] = $count;
        }
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Fetch total earnings from commissions for the current month
            $totalEarnings = Order::where('is_paid', 1)->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($order) {
                    // Calculate earnings: total_price * (commission / 100)
                    return $order->total_price * ($order->commission / 100);
                });

            // Store the total earnings in the corresponding month index
            $monthsCountEarnings[$month - 1] = $totalEarnings;
        }
        $totalEarnings = Order::where('is_paid', 1)->get()
            ->sum(function ($order) {
                // Calculate earnings: total_price * (commission / 100)
                return $order->total_price * ($order->commission / 100);
            });
        $data = [
            'status' => true,
            'total_users' => $totalUsers,
            'total_earnings' => round($totalEarnings, 2),
            'user_trend' => $monthlyTrend,
            'total_application' => $totalApplications,
            'application_trend' => $monthlyTrendApplication,
            'monthsCount' => $monthsCountUsers,
            'monthsCountEarnings' => $monthsCountEarnings,
        ];

        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }

    public function update(UserProfileUpdateRequest $request)
    {
        auth()->user()->first_name = $request->first_name;
        auth()->user()->last_name = $request->last_name;
        auth()->user()->phone = $request->phone;
        auth()->user()->dial_code = $request->dial_code;
        auth()->user()->country_code = $request->country_code;
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, User::IMAGES_DIRECTORY, 'local');
            auth()->user()->image = $saveFile['fileName'];
        }
        auth()->user()->save();
        $user = User::find(auth()->user()->id);

        return $this->response->success(
            new UserListingResource($user)
        );
        // return response()->json(['status' => true, 'message' => 'Profile Updated Successfully', 'data' => auth()->user()]);
    }

    public function updatePassword(ChangePasswordRequest $request)
    {
        $user = $this->userService->updatePassword($request, auth()->user()->id);

        return $this->response->success(
            new DefaultResource(['message' => 'Password changed successfully!'])
        );
    }

    public function signup(UserRegisterRequest $request, UserService $userService)
    {
        $data = $userService->create($request, 'user');

        return $this->response->success(
            new UserListingResource($data)
        );
    }

    public function login(LoginRequest $request)
    {
        $data = $this->userService->checkLogin($request, 'user');

        return $this->response->success(
            $data
        );
    }

    public function reset(ResetPassword $request)
    {
        $this->userService->reset($request);

        return response()->json(['status' => true, 'message' => 'Code send to email']);
    }

    public function setPassword(UserSetPasswordRequest $request)
    {
        $this->userService->verifyAndUpdatePassword($request);

        return response()->json(['status' => true, 'message' => 'Password Changed']);
    }

    public function verifyCode(VerifyActivationCodeRequest $request)
    {
        return response()->json(['status' => true, 'message' => 'Code verified']);
    }

    public function sendVerificationCode(ResetPassword $request)
    {
        $user = $this->userService->getUserByEmail($request->email);
        $this->userService->sendAccountVerification($user);
    }

    public function logout(Request $request)
    {
        $user = Auth::user();
        $user->tokens()->where('id', $user->currentAccessToken()->id)->delete();

        return response()->json(['status' => true, 'message' => 'Logout Successful']);
    }

    public function delete(Request $request)
    {
        $user = Auth::user();

        if (! $user) {
            return response()->json([
                'status' => false,
                'message' => 'User not found or not authenticated.',
            ], 404);
        }

        // Soft delete the authenticated user
        $user->delete();

        return response()->json([
            'status' => true,
            'message' => 'User deleted successfully .',
        ]);
    }

    public function profile(Request $request)
    {
        $user = auth()->guard()->user(); // ->load(['education', 'skills', 'experience']);

        return $this->response->success(
            new UserListingResource($user)
        );
    }

    public function userProfile(Request $request, User $user)
    {
        return $this->response->success(
            new UserListingResource($user)
        );
    }

    public function profileWithDetails($id)
    {
        $user = User::with(['images', 'socialLinks', 'userAnswers', 'userAnswers.question.category'])->find($id);

        return $this->response->success(
            new UserListingResource($user)
        );
    }
}
