<?php

namespace App\Http\Controllers\Api;

use App\Exceptions\BadRequestException;
use App\Http\Controllers\MainApiController;
use App\Http\Requests\Appointment\CreateAppointmentRequest;
use App\Http\Requests\Appointment\CreateAppointmentWebRequest;
use App\Http\Resources\Appointment\AppointmentDetailsResource;
use App\Http\Resources\Appointment\AppointmentListingResource;
use App\Http\Resources\DefaultResource;
use App\Services\AppointmentService;
use App\Services\ChatService;
use App\Services\Common\UserService;
use App\Services\StripeService;
use App\Services\TimeSlotService;
use Hash;
use Illuminate\Auth\AuthenticationException;

class AppointmentController extends MainApiController
{
    protected $appointmentService;

    public function __construct(AppointmentService $appointmentService)
    {
        parent::__construct();
        $this->appointmentService = $appointmentService;
    }

    public function create(CreateAppointmentRequest $request, StripeService $stripeService, TimeSlotService $timeSlotService, ChatService $chatService)
    {
        // Resolve timeslot from consultant, date and time (no need for time_slot_id in request)
        $timeslot = $timeSlotService->resolveSlotFromTemplate(
            $request->consultant_id,
            $request->date,
            $request->time
        );
        $checkAppointment = $this->appointmentService->checkAppointment(auth()->user()->id, $timeslot);
        if ($checkAppointment) {
            throw new BadRequestException(
                'You have already booked an appointment at this time and date.'
            );
        }
        // Inject resolved timeslot id so existing service logic keeps working
        $request->merge(['time_slot_id' => $timeslot->id]);

        $appointment = $this->appointmentService->create($request);
        $chat = $chatService->createChatWithRequestAccept(auth()->user()->id, $request->consultant_id);
        $stripeCheckoutSession = $stripeService->createCheckoutSession($appointment->service->charges, auth()->user()->id, $appointment->id, $chat->id);

        return $this->response->success(
            new DefaultResource($stripeCheckoutSession)
        );
    }

    public function createWeb(CreateAppointmentWebRequest $request, StripeService $stripeService, TimeSlotService $timeSlotService, ChatService $chatService)
    {
        $userService = new UserService;
        $user = $userService->getUserByEmail($request->email);
        if ($user) {
            if ($user->type === 'user') {
                if (! Hash::check($request->password, $user->password)) {
                    throw new BadRequestException(
                        'The provided credentials are incorrect.'
                    );
                }
            } else {
                throw new AuthenticationException(
                    'This email is already registered to a consultant.'
                );
            }
        } else {
            $user = $userService->create($request);
        }
        $timeslot = $timeSlotService->resolveSlotFromTemplate(
            $request->consultant_id,
            $request->date,
            $request->time
        );
        $checkAppointment = $this->appointmentService->checkAppointment($user->id, $timeslot);
        if ($checkAppointment) {
            throw new BadRequestException(
                'You have already booked an appointment at this time and date.'
            );
        }

        $request->merge(['time_slot_id' => $timeslot->id]);

        $appointment = $this->appointmentService->createWeb($request, $user);
        $chat = $chatService->createChatWithRequestAccept($user->id, $request->consultant_id);
        $stripeCheckoutSession = $stripeService->createCheckoutSession($appointment->service->charges, $user->id, $appointment->id, $chat->id);

        return $this->response->success(
            new DefaultResource($stripeCheckoutSession)
        );
    }

    public function index()
    {
        $appointment = $this->appointmentService->userPaidAppointment(auth()->user()->id);

        return $this->response->success(
            AppointmentListingResource::collection($appointment)
        );
    }

    public function view($id, ChatService $chatService)
    {
        $appointment = $this->appointmentService->view($id);
        $chat = $chatService->getChatIfExists($appointment->user_id, $appointment->consultant_id);
        $appointment->chat_id = $chat?->id;

        return $this->response->success(
            new AppointmentDetailsResource($appointment)
        );
    }
}
