<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\MainApiController;
use App\Http\Requests\Service\CreateServiceRequest;
use App\Http\Requests\Service\UpdateServiceRequest;
use App\Http\Resources\Service\ServiceDetailResource;
use App\Http\Resources\Service\ServiceResource;
use App\Services\ServiceService;
use Illuminate\Http\Request;

class ServiceController extends MainApiController
{
    protected $serviceService;

    public function __construct(ServiceService $serviceService)
    {
        $this->serviceService = $serviceService;
        parent::__construct();
    }

    public function index(Request $request)
    {
        $services = $this->serviceService->index();
        return $this->response->success(
            ServiceResource::collection($services)
        );
    }

    public function consultantServices($consultant_id)
    {
        $services = $this->serviceService->consultantServices($consultant_id);
        return $this->response->success(
            ServiceResource::collection($services)
        );
    }

    public function consultantServicesWithPagination($consultant_id)
    {
        $services = $this->serviceService->consultantServicesWithPagination($consultant_id);
        return $this->response->success(
            ServiceResource::collection($services)
        );
    }

    public function create(CreateServiceRequest $request)
    {
        $service = $this->serviceService->create($request);
        return $this->response->success(
            new ServiceDetailResource($service)
        );
    }

    public function show($id)
    {
        $service = $this->serviceService->show($id);
        return $this->response->success(
            new ServiceDetailResource($service)
        );
    }

    public function status($id)
    {
        $service = $this->serviceService->accountStatus($id);
        return $this->response->success(
            new ServiceDetailResource($service)
        );
    }

    public function update(UpdateServiceRequest $request, $id)
    {
        $serviceCategory = $this->serviceService->update($request, $id);
        return $this->response->success(
            new ServiceDetailResource($serviceCategory)
        );
    }

    public function delete($id)
    {
        $this->serviceService->delete($id);

        return $this->response->success([
            'message' => 'Service deleted successfully',
        ]);
    }

    public function reorder(Request $request)
    {
        $validated = $request->validate([
            'services' => 'required|array',
            'services.*.id' => 'required|integer|exists:services,id',
            'services.*.index' => 'required|integer|min:0',
        ]);

        $services = $this->serviceService->reorder($validated['services']);

        return $this->response->success(
            ServiceResource::collection($services)
        );
    }
}
