<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\User\LoginRequest;
use App\Http\Resources\Admin\UserListingResource;
use App\Models\Appointment;
use App\Models\Banner;
use App\Models\Order;
use App\Models\PaymentLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Services\AdminService;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\DefaultResource;
use App\Http\Controllers\MainApiController;
use App\Http\Requests\VerifyActivationCodeRequest;
use App\Http\Requests\Password\{ResetPassword, SetNewPasswordRequest, ChangePasswordRequest};
use App\Http\Requests\UpdateAdminProfileRequest;
use App\Models\User;
use Illuminate\Support\Facades\Auth;

class AuthController extends MainApiController
{
    private $adminService;

    public function __construct(AdminService $adminService)
    {
        parent::__construct();
        $this->adminService = $adminService;
    }


    public function update(UpdateAdminProfileRequest $request)
    {
        auth()->user()->first_name = $request->first_name;
        auth()->user()->last_name = $request->last_name;
        auth()->user()->phone = $request->phone;
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, User::IMAGES_DIRECTORY, 'local');
            auth()->user()->image = $saveFile['fileName'];
        }
        auth()->user()->save();
        return $this->response->success(
            new DefaultResource(['message' => 'Profile updated successfully!', 'data' => new UserListingResource(auth()->user())])
        );
        // return response()->json(['status' => true, 'message' => 'Profile Updated Successfully', 'data' => auth()->user()]);
    }


    public function updatePassword(ChangePasswordRequest $request)
    {
        $user = $this->adminService->updatePassword($request, auth()->user()->id);

        return $this->response->success(
            new DefaultResource(['message' => 'Password changed successfully!'])
        );
    }



    public function login(LoginRequest $request)
    {
        $data = $this->adminService->checkLogin($request, 'admin');
        return $this->response->success(
            new DefaultResource($data)
        );
    }

    public function reset(ResetPassword $request)
    {
        $response = $this->adminService->reset($request);
        if ($response) {
            return $this->response->success(
                new DefaultResource(['message' => 'We have sent you a verification code!'])
            );
        }
    }

    public function setPassword(SetNewPasswordRequest $request)
    {
        $data = $this->adminService->verifyAndUpdatePassword($request);
        return $this->response->success(
            new DefaultResource($data)
        );
    }

    public function verifyCode(VerifyActivationCodeRequest $request)
    {
        return $this->response->success(
            new DefaultResource(['message' => 'Code verified!'])
        );
    }

    public function sendVerificationCode(ResetPassword $request)
    {

        $user = $this->adminService->getUserByEmail($request->email);
        $this->adminService->sendAccountVerification($user);
        return $this->response->success(
            new DefaultResource(['message' => 'We have sent you a verification code!'])
        );
    }

    public function logout(Request $request)
    {
        $user = Auth::user();
        $user->tokens()->where('id', $user->currentAccessToken()->id)->delete();
        return $this->response->success(
            new DefaultResource(['message' => 'Logout Successful!'])
        );
    }

    public function profile(Request $request)
    {
        $admin = auth()->guard()->user();
        return $this->response->success(
            new DefaultResource($admin)
        );
    }

    public function dashboardStats(Request $request)
    {
        $totalUsers = User::where('type', 'user')->count();
        $totalConsultants = User::where('type', 'consultant')->count();
        $totalEarnings = PaymentLog::sum('amount');

        $newBookings = Appointment::where('status', 'upcoming')->count();
        // $totalBanners = Banner::count();
        // $totalOrders = Order::where('is_paid', 1)->count();
        $monthlyTrendApplication = User::selectRaw('COUNT(id) as count')->whereRaw("created_at LIKE '" . date('Y-m') . "%'")->first();
        $totalApplications = User::count();
        $monthlyTrendApplication = (isset($monthlyTrendApplication->count) && $monthlyTrendApplication->count > 0 ? round((($monthlyTrendApplication->count / $totalApplications) * 100), 2) : 0);
        $currentYear = request("year");
        // Prepare an array to hold the results
        $monthsCountUsers = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month
        $monthsCountEarnings = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month
        $monthsCountBookings = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month

        // Loop through each month from January to the current month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Count the number of users created in this month
            $count = User::where("type", 'user')->whereBetween('created_at', [$startOfMonth, $endOfMonth])->count();

            // Store the count in the corresponding month index (month - 1 because array index starts at 0)
            $monthsCountUsers[$month - 1] = $count;
        }
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Fetch total earnings from commissions for the current month
            $totalEarningsMonthly = PaymentLog::whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($paymentLog) {
                    // Calculate earnings: total_price * (commission / 100)
                    return $paymentLog->amount;
                });

            // Store the total earnings in the corresponding month index
            $monthsCountEarnings[$month - 1] = $totalEarningsMonthly;
        }
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Fetch total earnings from commissions for the current month
            $monthsCountBookingsCount = Appointment::whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->count();

            // Store the total earnings in the corresponding month index
            $monthsCountBookings[$month - 1] = $monthsCountBookingsCount;
        }
        $data = [
            'status' => true,
            'total_users' => $totalUsers,
            'total_consultants' => $totalConsultants,
            'total_earnings' => $totalEarnings,
            'new_bookings' => $newBookings,
            'monthly_users' => $monthsCountUsers,
            'monthly_earnings' => $monthsCountEarnings,
            'monthly_bookings' => $monthsCountBookings,
        ];
        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }

    public function chartEarningStats()
    {
        $currentYear = request("year");
        $monthsCountEarnings = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Fetch total earnings from commissions for the current month
            $totalEarningsMonthly = PaymentLog::whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($paymentLog) {
                    // Calculate earnings: total_price * (commission / 100)
                    return $paymentLog->amount;
                });

            // Store the total earnings in the corresponding month index
            $monthsCountEarnings[$month - 1] = $totalEarningsMonthly;
        }
        $data = [
            'status' => true,
            'monthly_earnings' => $monthsCountEarnings,

        ];
        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }

    public function chartUsersStats()
    {
        $currentYear = request("year");
        $monthsCountUsers = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Count the number of users created in this month
            $count = User::where("type", 'user')->whereBetween('created_at', [$startOfMonth, $endOfMonth])->count();

            // Store the count in the corresponding month index (month - 1 because array index starts at 0)
            $monthsCountUsers[$month - 1] = $count;
        }
        $data = [
            'status' => true,
            'monthly_users' => $monthsCountUsers,

        ];
        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }

    public function chartBookingStats()
    {
        $currentYear = request("year");
        $monthsCountBookings = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Fetch total earnings from commissions for the current month
            $monthsCountBookingsCount = Appointment::whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->count();

            // Store the total earnings in the corresponding month index
            $monthsCountBookings[$month - 1] = $monthsCountBookingsCount;
        }
        $data = [
            'status' => true,
            'monthly_bookings' => $monthsCountBookings,

        ];
        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }
}
